/**
* \file: handle_remount_request_cmd.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <poll.h>
#include <errno.h>

#include "handle_mnt_umnt_request_cmd/handle_remount_request_cmd.h"
#include "utils/path.h"
#include "automounter_api.h"
#include "automounter_api_events.h"
#include "automounter_api_ctrl.h"

//------------------------------------------ private attributes -------------------------------------------------------
static char* mount_point=NULL;
static const char* mount_options=NULL;

static bool waiting_for_request_done_signal=true;
static error_code_t request_result=RESULT_OK;

static automounter_api_callbacks_t cbs={0};
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------------ private function declarations --------------------------------------------
static error_code_t handle_remount_cmd_parse_args(int argc, char *argv[]);
static error_code_t handle_remount_cmd_start(void);
static void handle_remount_cmd_print_help(void);
static void handle_remount_cmd_print_usage(void);

static error_code_t handle_remount_cmd_init(void);
static void handle_remount_cmd_deinit(void);

static void handle_remount_done_callback(int request_id, error_code_t result, const char *error_message);
static void handle_remount_cmd_on_connection_lost(void);
static error_code_t handle_remount_cmd_mainloop(void);
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------------ vtable for this command --------------------------------------------------
automounterctl_command_vtable_t handle_remount_request_cmd_vtable =
{
		.command_description =
				"Triggers the automounter to remount a partition with given options.",
				.command = "remount",
				.init = handle_remount_cmd_init,
				.parse_args = handle_remount_cmd_parse_args,
				.start = handle_remount_cmd_start,
				.deinit = handle_remount_cmd_deinit
};
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------------ private function definitions ---------------------------------------------
static error_code_t handle_remount_cmd_parse_args(int argc, char *argv[])
{
	//no additional parameter given, we are missing something
	if (argc == 2)
	{
		printf("Too few arguments. A mount point and the mount options must be passed.\n");
		handle_remount_cmd_print_usage();
		return RESULT_HELP_PRINTED;
	}

	//we can be sure that we have more than 2 parameters (automounterctl <command>) passed,
	//otherwise we would have never got here.
	if (strcmp(argv[2], "--help") == 0 || strcmp(argv[2], "-h") == 0)
	{
		handle_remount_cmd_print_help();
		return RESULT_HELP_PRINTED;
	}

	//no help requested. At this point we expect to have to additional parameters (mountpoint, options)
	if (argc<4)
	{
		printf("Too few arguments. A mount point and the mount options must be passed.\n");
		handle_remount_cmd_print_usage();
		return RESULT_HELP_PRINTED;
	}

	//now we are sure to have enough arguments
	mount_point=argv[2];
	mount_options=argv[3];

	//in case remove the trailing slash
	path_remove_trailing_slash(mount_point);

	return RESULT_OK;
}

static void handle_remount_cmd_print_help(void)
{
	printf("\n");
	printf("Automounter Control Utility - Used to control the ADIT automounter daemon.\n\n");
	printf("automounterctl %s <mount point> <mount options> | --help\n",
			handle_remount_request_cmd_vtable.command);
	printf("\t-h,--help:\tdisplays this help and exits.\n\n");
	printf("This command triggers the automounter daemon to remount a given <mount point> \n"
			"with given <mount options>. Mount points that have not been mounted by the\n"
			"automounter before are not touched.\n\n");
}

static void handle_remount_cmd_print_usage(void)
{
	printf("\n");
	printf("Usage: automounterctl %s <mount point> <mount options> | --help\n",
			handle_remount_request_cmd_vtable.command);
	printf("\t-h,--help:\t\tdisplays this help and exits.\n");
}

static error_code_t handle_remount_cmd_start(void)
{
	error_code_t result;
	result=automounter_api_connect();

	waiting_for_request_done_signal=true;
	request_result=RESULT_OK;

	if (result == RESULT_OK)
		result = automounter_api_remount_partition_by_mountpoint(mount_point,mount_options,-1,
				handle_remount_done_callback);

	if (result != RESULT_OK)
	{
		printf("Error executing the remount command.\n");
		return result;
	}


	result=handle_remount_cmd_mainloop();

	//check if we had an error causing us to leave the mainloop
	if (result!=RESULT_OK)
	{
		printf("Connection to the automounter got lost while processing the request.\n");
		return result;
	}

	// we got an answer. Check the results transmitted from the automounter
	if (request_result==RESULT_NOT_MOUNTED)
		printf("The partition that belongs to given mount point has been unmounted before.\n");
	else if (request_result==RESULT_INVALID)
		printf("Given mount point not found or not under the responsibility of the automounter.\n");
	else if (request_result==RESULT_NORESOURCE)
		printf("We run into resource issues. Memory problems?\n");

	return request_result;
}

static error_code_t handle_remount_cmd_init(void)
{
	error_code_t result;

	result=automounter_api_init("automounterctl", LOGGER_LEVEL_ERROR, true);
	if (result==RESULT_OK)
	{
		cbs.on_connection_lost=handle_remount_cmd_on_connection_lost;
		automounter_api_register_callbacks(&cbs);
	}

	return result;
}

static void handle_remount_cmd_deinit(void)
{
	automounter_api_disconnect();
	automounter_api_deinit();
}

static void handle_remount_done_callback(int request_id, error_code_t result, const char *error_message)
{
	//we passed -1 here because we are not sending more than one request at a time. So we know from which
	//request the answer comes.
	(void)request_id;
	//we are extracting the message from the result code for now
	(void)error_message;
	waiting_for_request_done_signal=false;
	request_result=result;
}

static void handle_remount_cmd_on_connection_lost(void)
{
	printf("Connection to the automounter got lost unexpectedly.\n");
	request_result=RESULT_SOCKET_ERR;
	waiting_for_request_done_signal=false;
}

static error_code_t handle_remount_cmd_mainloop(void)
{
	error_code_t result=RESULT_OK;
	int am_pollfd;
	struct pollfd am_pollfd_struc;

	am_pollfd=automounter_api_get_pollfd();
	if (am_pollfd==-1)
		return RESULT_INVALID;

	am_pollfd_struc.fd=am_pollfd;
	am_pollfd_struc.events=POLLIN;

	//we sent the command successfully, now we are waiting for a response
	while(result == RESULT_OK && waiting_for_request_done_signal)
	{
		if (poll(&am_pollfd_struc,1,-1)<=0)
		{
			if (errno!=EINTR)
			{
				printf("ERROR: Automounterctl detected problems with its poll file descriptor in the main loop.\n");
				result=RESULT_NORESOURCE;
			}
			else
				waiting_for_request_done_signal=false;
			continue;
		}
		automounter_api_dispatch_event();
	}

	return result;
}
//---------------------------------------------------------------------------------------------------------------------
